/*************************************************************************************
  * File Name       : system_stm32f0xx.c
  * Description     : This source file contains the System Clock configurations
  ************************************************************************************/

/*************************************************************************************
 * SystemInit():	Sets up System Clock source, PLL Multiplier &	Divider factors,
 * 								as well as AHB/APBx prescalers and Flash settings.
 * 								This function is called at startup from 'startup_stm32f0xx.s'.
 *
 * SystemCoreClock variable:	Contains core clock (HCLK) & can be used to setup the
 * 														SysTick Timer or configure other parameters.
 *
 * SystemCoreClockUpdate(): Updates variable SystemCoreClock. Must be called whenever
 * 													core clock is changed during program execution.
 *
 * 1.	HSI (8 MHz) is used as system clock source after each reset. 'SystemInit()' is
 * 		called from 'startup_stm32f0xx.s' to configure system clock before branch to main.
 *
 * 2.	If selected system clock source fails to startup, 'SystemInit()' does nothing &
 * 		HSI continues to be used as system clock source. Note: Code can be added inside
 * 		'SetSysClock()' to handle this situation.
 *
 * 3.	HSE crystal default is 8MHz (see 'HSE_VALUE' define in "stm32f0xx.h'.
 *		Note: If HSE is used as system clock source, directly or via PLL, 'HSE_VALUE'
 *					must be changed accordingly.
 ***************************************************************************************/

/* ======================================================== *
 *    System Clock source					|				PLL (HSE)					*
 * --------------------------------------------------------	*
 *        SYSCLK(Hz)              |  			48000000					*
 * --------------------------------------------------------	*
 *        HCLK(Hz)                | 			48000000					*
 * --------------------------------------------------------	*
 *        AHB Prescaler           | 					1							*
 * --------------------------------------------------------	*
 *        APB Prescaler           | 					1							*
 * --------------------------------------------------------	*
 *        HSE Frequency(Hz)       | 			32000000					*
 * --------------------------------------------------------	*
 *        PLLMUL                  | 					3							*
 * --------------------------------------------------------	*
 *        PREDIV                  | 					2							*
 * --------------------------------------------------------	*
 *        Flash Latency(WS)       |						1							*
 * --------------------------------------------------------	*
 *        Prefetch Buffer         |						ON						*
 * ======================================================== */

/** @addtogroup CMSIS
  * @{
  */

/** @addtogroup stm32f0xx_system
  * @{
  */  
  
/** @addtogroup STM32F0xx_System_Private_Includes
  * @{
  */

#include "stm32f0xx.h"

/**
  * @}
  */

/** @addtogroup STM32F0xx_System_Private_TypesDefinitions
  * @{
  */

/**
  * @}
  */

/** @addtogroup STM32F0xx_System_Private_Defines
  * @{
  */
/**
  * @}
  */

/** @addtogroup STM32F0xx_System_Private_Macros
  * @{
  */

/**
  * @}
  */

/** @addtogroup STM32F0xx_System_Private_Variables
  * @{
  */
uint32_t SystemCoreClock    = 48000000;
__I uint8_t AHBPrescTable[16] = {0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 6, 7, 8, 9};

/**
  * @}
  */

/** @addtogroup STM32F0xx_System_Private_FunctionPrototypes
  * @{
  */

static void SetSysClock(void);

/**
  * @}
  */

/** @addtogroup STM32F0xx_System_Private_Functions
  * @{
  */

void SystemInit (void)
{    
  /* Set HSION bit */
  RCC->CR |= (uint32_t)0x00000001;

  /* Reset SW[1:0], HPRE[3:0], PPRE[2:0], ADCPRE and MCOSEL[2:0] bits */
  RCC->CFGR &= (uint32_t)0xF8FFB80C;
  
  /* Reset HSEON, CSSON and PLLON bits */
  RCC->CR &= (uint32_t)0xFEF6FFFF;

  /* Reset HSEBYP bit */
  RCC->CR &= (uint32_t)0xFFFBFFFF;

  /* Reset PLLSRC, PLLXTPRE and PLLMUL[3:0] bits */
  RCC->CFGR &= (uint32_t)0xFFC0FFFF;

  /* Reset PREDIV1[3:0] bits */
  RCC->CFGR2 &= (uint32_t)0xFFFFFFF0;

  /* Reset USARTSW[1:0], I2CSW, CECSW and ADCSW bits */
  RCC->CFGR3 &= (uint32_t)0xFFFFFEAC;

  /* Reset HSI14 bit */
  RCC->CR2 &= (uint32_t)0xFFFFFFFE;

  /* Disable all interrupts */
  RCC->CIR = 0x00000000;

  /* Configure the System clock frequency, AHB/APBx prescalers and Flash settings */
  SetSysClock();
}

/**************************************************************************************
 * SystemCoreClock contains core clock (HCLK) & can be used to setup SysTick timer.		*
 * Call this function each time HCLK changes to update SystemCoreClock variable.  		*
 * 																																										*
 * If SYSCLK source is HSI, SystemCoreClock contains HSI_VALUE (see stm32f0xx.h)			*
 * 																																										*
 * If SYSCLK source is HSE, SystemCoreClock contains HSE_VALUE (see stm32f0xx.h)			*
 * 																																										*
 * If SYSCLK source is PLL, SystemCoreClock contains HSE_VALUE OR											*
 * HSI_VALUE, multiplied/divided by the PLL factors (see stm32f0xx.h).								*
 **************************************************************************************/

void SystemCoreClockUpdate (void)
{
  uint32_t tmp = 0, pllmull = 0, pllsource = 0, prediv1factor = 0;

  /* Get SYSCLK source -------------------------------------------------------*/
  tmp = RCC->CFGR & RCC_CFGR_SWS;
  
  switch (tmp)
  {
    case 0x00:  /* HSI used as system clock */
      SystemCoreClock = HSI_VALUE;
      break;
    case 0x04:  /* HSE used as system clock */
      SystemCoreClock = HSE_VALUE;
      break;
    case 0x08:  /* PLL used as system clock */
      /* Get PLL clock source and multiplication factor ----------------------*/
      pllmull = RCC->CFGR & RCC_CFGR_PLLMULL;
      pllsource = RCC->CFGR & RCC_CFGR_PLLSRC;
      pllmull = ( pllmull >> 18) + 2;
      
      if (pllsource == 0x00)
      {
        /* HSI oscillator clock divided by 2 selected as PLL clock entry */
        SystemCoreClock = (HSI_VALUE >> 1) * pllmull;
      }
      else
      {
        prediv1factor = (RCC->CFGR2 & RCC_CFGR2_PREDIV1) + 1;
        /* HSE oscillator clock selected as PREDIV1 clock entry */
        SystemCoreClock = (HSE_VALUE / prediv1factor) * pllmull; 
      }      
      break;
    default: /* HSI used as system clock */
      SystemCoreClock = HSI_VALUE;
      break;
  }
  /* Calculate HCLK clock frequency ----------------*/
  /* Get HCLK prescaler */
  tmp = AHBPrescTable[((RCC->CFGR & RCC_CFGR_HPRE) >> 4)];
  /* HCLK clock frequency */
  SystemCoreClock >>= tmp;  
}

/* Configure System clock frequency, AHB/APBx prescalers & Flash */

static void SetSysClock(void)
{
  __IO uint32_t StartUpCounter = 0, HSEStatus = 0;

/******************************************************************************/
/*            PLL (clocked by HSE) - Used as System clock source                */
/******************************************************************************/

  /* SYSCLK, HCLK, PCLK configuration ----------------------------------------*/
  /* Enable HSE */    
  RCC->CR |= ((uint32_t)RCC_CR_HSEON);

  /* Wait till HSE is ready and if Time out is reached exit */
  do
  {
    HSEStatus = RCC->CR & RCC_CR_HSERDY;
    StartUpCounter++;  
  } while((HSEStatus == 0) && (StartUpCounter != HSE_STARTUP_TIMEOUT));

  if ((RCC->CR & RCC_CR_HSERDY) != RESET)
  {
    HSEStatus = (uint32_t)0x01;
  }
  else
  {
    HSEStatus = (uint32_t)0x00;
  }  

  if (HSEStatus == (uint32_t)0x01)
  {
    /* Enable Prefetch Buffer and set Flash Latency */
    FLASH->ACR = FLASH_ACR_PRFTBE | FLASH_ACR_LATENCY;

     /* HCLK = SYSCLK / 1 */
     RCC->CFGR |= (uint32_t)RCC_CFGR_HPRE_DIV1;
       
     /* PCLK = HCLK / 1 */
     RCC->CFGR |= (uint32_t)RCC_CFGR_PPRE_DIV1;

    /* PLL configuration */
    RCC->CFGR &= (uint32_t)((uint32_t)~(RCC_CFGR_PLLSRC | RCC_CFGR_PLLXTPRE | RCC_CFGR_PLLMULL));
    RCC->CFGR |= (uint32_t)(RCC_CFGR_PLLSRC_PREDIV1 | RCC_CFGR_PLLMULL3);
    RCC->CFGR2 &= (uint32_t)((uint32_t)~(RCC_CFGR2_PREDIV1));
    RCC->CFGR2 |= (uint32_t)(RCC_CFGR2_PREDIV1_DIV2);

    /* Enable PLL */
    RCC->CR |= RCC_CR_PLLON;

    /* Wait till PLL is ready */
    while((RCC->CR & RCC_CR_PLLRDY) == 0)
    {
    }

    /* Select PLL as system clock source */
    RCC->CFGR &= (uint32_t)((uint32_t)~(RCC_CFGR_SW));
    RCC->CFGR |= (uint32_t)RCC_CFGR_SW_PLL;    

    /* Wait till PLL is used as system clock source */
    while ((RCC->CFGR & (uint32_t)RCC_CFGR_SWS) != (uint32_t)RCC_CFGR_SWS_PLL)
    {
    }
  }
  else
  { /* If HSE fails to start-up, the application will have incorrect clock
       configuration. Add code here to deal with this error */
  }
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
