/********************************************************************************
  * File Name          : gpio.c
  * Description        : This source file contains the General Purpose I/O code
  *******************************************************************************/

/* ============================== Includes ============================== */
#include "stm32f0xx.h"
#include "main.h"
#include "timer.h"
#include "gpio.h"


/* ========================= External Variables ========================= */


/* =========================== Private Defines =========================== */
#define PORTA_INPUTS				(START_SW | H2O_SW | LID_SW | TEST_PIN_SW) //Circor added "|TEST_PIN_SW"
#define PORTA_OUTPUTS				(WASH_LED | SPIN_LED)

#define PORTB_OUTPUTS				(L_MAST_CTRL | MOT_FWD_CTRL | MOT_REV_CTRL | H2O_COLD_CTRL | RELAY_CTRL)

#define SW_DEBOUNCE_PERIOD			20
#define START_SW_CHK_DELAY			20
#define H2O_SW_TRIG_MAX_WAIT		1500
#define LID_CLS_SW_DELAY				100
#define PWR_SW_DELAY						50

/* ========================== Private Variables ========================== */
uint16_t PwrOnDlyCnt = 0;
uint16_t MotFwdOnCnt = 0;
uint16_t MotFwdDlyOnCnt = 0;
uint16_t MotRevOnCnt = 0;
uint16_t MotRevDlyOnCnt = 0;
uint16_t H2OOnCnt = 0;
uint16_t H2ODlyOnCnt = 0;
uint16_t H2OOnTimeoutCnt = 0;

uint8_t bitStatus = 0;
uint8_t tempMotState = 0;
uint8_t tempWaterState = 0;
uint16_t tempPwrOnDlyCnt = 0;
uint16_t tempMotFwdOnCnt = 0;
uint16_t tempMotFwdDlyOnCnt = 0;
uint16_t tempMotRevOnCnt = 0;
uint16_t tempMotRevDlyOnCnt = 0;
uint16_t tempH2OOnCnt = 0;
uint16_t tempH2ODlyOnCnt = 0;
uint16_t tempH2OOnTimeoutCnt = 0;
uint32_t tempSysTickDlyCnt = 0;


/* ===================== Private Function Prototypes ====================== */
void gpio_InitVars(void);
void gpio_ConfigPortAInputs(void);
void gpio_ConfigPortAOutputs(void);
void gpio_ConfigPortBOutputs(void);
void gpio_SysIdleInit(void);
void gpio_LedCtrl(void);


/* ============================== Functions =============================== */
void gpio_InitVars(void)
{
	sSysFlags.StartSwFlag = RESET;
	sSysFlags.LidSwFlag = RESET;
	sWashLed.LedState = LED_OFF;
	sSpinLed.LedState = LED_OFF;
	sSysStatus.MotState = MOT_OFF;
	sSysStatus.H2OState = H2O_OFF;
}

void gpio_ConfigPortAInputs(void)
{
	GPIO_InitTypeDef GPIO_InitStruct;
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_GPIOA, ENABLE);

  GPIO_InitStruct.GPIO_Pin = PORTA_INPUTS;
  GPIO_InitStruct.GPIO_Mode = GPIO_Mode_IN;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PuPd_UP;
  GPIO_InitStruct.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_Init(GPIOA, &GPIO_InitStruct);
}

void gpio_ConfigPortAOutputs(void)
{
	GPIO_InitTypeDef GPIO_InitStruct;
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_GPIOA, ENABLE);

  GPIO_InitStruct.GPIO_Pin = PORTA_OUTPUTS;
  GPIO_InitStruct.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStruct.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_InitStruct.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_Init(GPIOA, &GPIO_InitStruct);

	GPIO_ResetBits(GPIOA, PORTA_OUTPUTS);
}

void gpio_ConfigPortBOutputs(void)
{
	GPIO_InitTypeDef GPIO_InitStruct;
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_GPIOB, ENABLE);

	GPIO_InitStruct.GPIO_Pin = PORTB_OUTPUTS;
  GPIO_InitStruct.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStruct.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_InitStruct.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_Init(GPIOB, &GPIO_InitStruct);

	GPIO_ResetBits(GPIOB, PORTB_OUTPUTS);
}

void gpio_ConfigAll(void)
{
	gpio_ConfigPortAInputs();
	gpio_ConfigPortAOutputs();
	gpio_ConfigPortBOutputs();
	gpio_InitVars();
	gpio_LedCtrl();
}

void gpio_LedCtrl(void)
{
	static uint8_t washLedState = LED_OFF;
	static uint8_t spinLedState = LED_OFF;

	if (sWashLed.LedState != washLedState)
	{
		washLedState = sWashLed.LedState;
		GPIO_WriteBit(WASH_LED_PORT, WASH_LED, sWashLed.LedState);
	}

	if (sSpinLed.LedState != spinLedState)
	{
		spinLedState = sSpinLed.LedState;
		GPIO_WriteBit(SPIN_LED_PORT, SPIN_LED, sSpinLed.LedState);
	}
}

void gpio_MotOffCtrl(void)
{
	sSysStatus.MotState = MOT_OFF;
	time_SysTickDelay(PWR_SW_DELAY);
	gpio_LidSwTrig();
	GPIO_WriteBit(MOT_CTRL_PORT, L_MAST_CTRL, OFF);
	time_SysTickDelay(PWR_SW_DELAY);
	gpio_LidSwTrig();
	GPIO_WriteBit(MOT_CTRL_PORT, RELAY_CTRL, OFF);
}

void gpio_MotFwdCtrl(void)
{
	sSysStatus.MotState = MOT_FWD;
	time_SysTickDelay(PWR_SW_DELAY);
	gpio_LidSwTrig();
	GPIO_WriteBit(MOT_CTRL_PORT, L_MAST_CTRL, OFF);
	time_SysTickDelay(PWR_SW_DELAY);
	gpio_LidSwTrig();
	GPIO_WriteBit(MOT_CTRL_PORT, RELAY_CTRL, OFF);
	time_SysTickDelay(PWR_SW_DELAY);
	gpio_LidSwTrig();
	GPIO_WriteBit(MOT_CTRL_PORT, L_MAST_CTRL, ON);
	MotFwdOnCnt = 0;
}

void gpio_MotRevCtrl(void)
{
	sSysStatus.MotState = MOT_REV;
	time_SysTickDelay(PWR_SW_DELAY);
	gpio_LidSwTrig();
	GPIO_WriteBit(MOT_CTRL_PORT, L_MAST_CTRL, OFF);
	time_SysTickDelay(PWR_SW_DELAY);
	gpio_LidSwTrig();
	GPIO_WriteBit(MOT_CTRL_PORT, RELAY_CTRL, ON);
	time_SysTickDelay(PWR_SW_DELAY);
	gpio_LidSwTrig();
	GPIO_WriteBit(MOT_CTRL_PORT, L_MAST_CTRL, ON);
	sSpinLed.LedState = LED_ON;
	gpio_LedCtrl();
	MotRevOnCnt = 0;
}

void gpio_H2OCtrl(uint8_t waterState)
{
	switch (waterState)
	{
		case H2O_OFF:
			GPIO_WriteBit(H2O_CTRL_PORT, H2O_COLD_CTRL, OFF);
			sSysStatus.H2OState = H2O_OFF;
			break;

  	case H2O_COLD:
  		GPIO_WriteBit(H2O_CTRL_PORT, H2O_COLD_CTRL, ON);
  		sSysStatus.H2OState = H2O_COLD;
  		H2OOnTimeoutCnt = 0;
  		break;

  	default:
  		break;
  }
}

void gpio_StartSwWait(void)
{
	uint8_t startSw = SET;

	while (startSw == SET)
	{
		time_SysTickDelay(START_SW_CHK_DELAY);
		startSw = GPIO_ReadInputDataBit(SWs_PORT, START_SW);
		if (startSw == RESET)
		{
			time_SysTickDelay(SW_DEBOUNCE_PERIOD);
			gpio_LidSwTrig();
			startSw = GPIO_ReadInputDataBit(SWs_PORT, START_SW);
			if (startSw == RESET)
			{
				gpio_MotOffCtrl();
				gpio_H2OCtrl(H2O_COLD);
				sWashLed.LedState = LED_ON;
				sSpinLed.LedState = LED_OFF;
				gpio_LedCtrl();
				break;
			}
			startSw = SET;
		}
	}
}

uint8_t gpio_H2OPressSwTrigWait(void)
{
	uint8_t h20SwPin = SET;
	uint16_t pressSwMaxWait = 0;

	while ( (h20SwPin == SET) && (pressSwMaxWait++ < H2O_SW_TRIG_MAX_WAIT) )
	{
		h20SwPin = GPIO_ReadInputDataBit(SWs_PORT, H2O_SW);
		if (h20SwPin == RESET)
		{
			time_SysTickDelay(SW_DEBOUNCE_PERIOD);
			gpio_LidSwTrig();
			h20SwPin = GPIO_ReadInputDataBit(SWs_PORT, H2O_SW);
			if (h20SwPin == RESET)
			{
				break;
			}
		}
		time_SysTickDelay(SW_DEBOUNCE_PERIOD);
		gpio_LidSwTrig();
	}

	gpio_H2OCtrl(H2O_OFF);
	if (pressSwMaxWait >= H2O_SW_TRIG_MAX_WAIT)
	{
		h20SwPin = SET;
	}
	return h20SwPin;
}

void gpio_LidSwChk(void)
{
	uint8_t lidSwState = LID_CLS;
	uint32_t delayCntr = 0;

	if (LidSwChkTimeCnt >= LID_SW_CHK_PERIOD)
	{
		LidSwChkTimeCnt = 0;
		lidSwState = GPIO_ReadInputDataBit(SWs_PORT, LID_SW);
		if (lidSwState == LID_OPEN)
		{
			delayCntr = WHILE_LOOP_200ms_DLY;
			while (delayCntr-- < 0);
			lidSwState = GPIO_ReadInputDataBit(SWs_PORT, LID_SW);
			if (lidSwState == LID_OPEN)
			{
				tempSysTickDlyCnt = SysTickDlyCnt;
				SysTickDlyCnt = 0;
				sSysFlags.LidSwFlag = SET;
			}
		}
	}
}

void gpio_LidSwTrig(void)
{
	uint8_t lidSwState = LID_CLS;
	uint32_t delayCntr = 0;

	if (sSysFlags.LidSwFlag == SET)
	{
		lidSwState = LID_OPEN;
		tempMotState = sSysStatus.MotState;
		tempWaterState = sSysStatus.H2OState;
		tempPwrOnDlyCnt = PwrOnDlyCnt;
		tempMotFwdOnCnt = MotFwdOnCnt;
		tempMotFwdDlyOnCnt = MotFwdDlyOnCnt;
		tempMotRevOnCnt = MotRevOnCnt;
		tempMotRevDlyOnCnt = MotRevDlyOnCnt;
		tempH2OOnCnt = H2OOnCnt;
		tempH2ODlyOnCnt = H2ODlyOnCnt;
		tempH2OOnTimeoutCnt = H2OOnTimeoutCnt;

		gpio_H2OCtrl(H2O_OFF);
		sSysStatus.MotState = MOT_OFF;
		GPIO_WriteBit(MOT_CTRL_PORT, L_MAST_CTRL, OFF);
		GPIO_WriteBit(MOT_CTRL_PORT, RELAY_CTRL, OFF);

		while (lidSwState == LID_OPEN)
		{
			lidSwState = GPIO_ReadInputDataBit(SWs_PORT, LID_SW);
			if (lidSwState == LID_CLS)
			{
				delayCntr = WHILE_LOOP_200ms_DLY;
				while (delayCntr-- > 0);
				lidSwState = GPIO_ReadInputDataBit(SWs_PORT, LID_SW);
				if (lidSwState == LID_CLS)
				{
					break;
				}
			}
		}

		sSysStatus.MotState = tempMotState;
		sSysStatus.H2OState = tempWaterState;

		if (sSysStatus.MotState != MOT_OFF)
		{
			delayCntr = WHILE_LOOP_500ms_DLY;
			while (delayCntr-- > 0);
			if (sSysStatus.MotState == MOT_REV)
			{
				GPIO_WriteBit(MOT_CTRL_PORT, RELAY_CTRL, ON);
				delayCntr = WHILE_LOOP_500ms_DLY;
				while (delayCntr-- > 0);
			}
			GPIO_WriteBit(MOT_CTRL_PORT, L_MAST_CTRL, ON);
		}
		gpio_H2OCtrl(sSysStatus.H2OState);

		PwrOnDlyCnt = tempPwrOnDlyCnt;
		MotFwdOnCnt = tempMotFwdOnCnt;
		MotFwdDlyOnCnt = tempMotFwdDlyOnCnt;
		MotRevOnCnt = tempMotRevOnCnt;
		MotRevDlyOnCnt = tempMotRevDlyOnCnt;
		H2OOnCnt = tempH2OOnCnt;
		H2ODlyOnCnt = tempH2ODlyOnCnt;
		H2OOnTimeoutCnt = tempH2OOnTimeoutCnt;
		SysTickDlyCnt = tempSysTickDlyCnt;
		sSysFlags.LidSwFlag = RESET;
		LidSwChkTimeCnt = 0;
	}
}

void gpio_ToggleBit(GPIO_TypeDef* GPIOx, uint16_t GPIO_Pin)
{
  assert_param(IS_GPIO_PIN(GPIO_Pin));
  GPIOx->ODR ^= GPIO_Pin;
}

void gpio_MotH2OCtrl(void)
{
  uint8_t h2oFlag = OFF;
  uint8_t motFlag = OFF;

  H2ODlyOnCnt = 0;
  MotRevDlyOnCnt = 0;

  while ( (h2oFlag == OFF) || (motFlag == OFF) )
  {
    if (h2oFlag == OFF)
    {
      if ( (sSysStatus.H2OState == H2O_OFF) && (H2ODlyOnCnt >= sMotH2OCtrl.H2ODlyOn) )
      {
      	H2OOnCnt = 0;
      	gpio_H2OCtrl(H2O_COLD);
      }
      else if ( (sSysStatus.H2OState == H2O_COLD) && (H2OOnCnt >= sMotH2OCtrl.H2OOnTime) )
      {
      	h2oFlag = ON;
      	gpio_H2OCtrl(H2O_OFF);
      }
    }

    if (motFlag == OFF)
    {
      if ( (MotRevDlyOnCnt >= sMotH2OCtrl.MotDlyOn) && (sSysStatus.MotState == MOT_OFF) )
      {
        MotRevOnCnt = 0;
        gpio_MotRevCtrl();
      }
      else if ( (MotRevOnCnt >= sMotH2OCtrl.MotOnTime) && (sSysStatus.MotState == MOT_REV) )
      {
        motFlag = ON;
        gpio_MotOffCtrl();
      }
    }
    gpio_LidSwChk();
    gpio_LidSwTrig();
  }
}

void gpio_SysIdleInit(void)
{
	sSysFlags.StartSwFlag = RESET;
	sWashLed.LedState = LED_OFF;
	sSpinLed.LedState = LED_OFF;
	gpio_LedCtrl();
}

void gpio_SysCtrlSeq(void)
{

	sWashLed.LedState = LED_ON; //FOR TESTING THE PROGRAMM
	sSpinLed.LedState = LED_ON;


	uint8_t h2oPressSwStatus = SET;

	gpio_StartSwWait();
	h2oPressSwStatus = gpio_H2OPressSwTrigWait();
	if (h2oPressSwStatus == SET)
	{
		return;
	}
  MotFwdDlyOnCnt = 0;
  while (MotFwdDlyOnCnt < H2O_OFF_DLY_MOT_FWD)
  {
  	time_SysTickDelay(LID_SW_CHK_PERIOD);
  	gpio_LidSwTrig();
  }
  gpio_MotFwdCtrl();
 	gpio_LidSwTrig();
        //Circor: Change the value of (MOT_FWD_ON_TIME_1) to reduce testing time __________________________________*(0v0)*___________
	
	int16_t mot_fwd_on_time_one = MOT_FWD_ON_TIME_1; //720
	int16_t test_mode_switch = GPIO_ReadInputDataBit(SWs_PORT, TEST_PIN_SW);
	if(test_mode_switch == RESET) mot_fwd_on_time_one = 8;
	

  while (MotFwdOnCnt < mot_fwd_on_time_one)                 //720 changed MOT_FWD_ON_TIME_1 to mot_fwd_on_time_one
  {
  	gpio_LidSwChk();
  	gpio_LidSwTrig();
  }
  gpio_MotOffCtrl();
	//Circor: Change these variables to reduce testing time // ________________________________________________*(0v0)*___________
  if(test_mode_switch == SET) {
	sMotH2OCtrl.MotDlyOn = MOT_REV_DLY_ON_TIME_1;     // 4
	sMotH2OCtrl.MotOnTime = MOT_REV_ON_TIME_1;        // 300
	sMotH2OCtrl.H2ODlyOn = MOT_REV_H2O_DLY_ON_TIME_1; //90
	sMotH2OCtrl.H2OOnTime = H2O_ON_TIME_1;            //30  
	gpio_MotH2OCtrl();
	sMotH2OCtrl.MotDlyOn = MOT_REV_DLY_ON_TIME_2;     //4
	sMotH2OCtrl.MotOnTime = MOT_REV_ON_TIME_2;        //300
	sMotH2OCtrl.H2ODlyOn = MOT_REV_H2O_DLY_ON_TIME_2; //4
	sMotH2OCtrl.H2OOnTime = H2O_ON_TIME_2;            //30
	gpio_MotH2OCtrl();
	gpio_SysIdleInit();
	}
	//Circor: Shorter delays for testmode only
  else if(test_mode_switch == RESET) { 
	sMotH2OCtrl.MotDlyOn = MOT_REV_DLY_ON_TIME_1;     // 4
	sMotH2OCtrl.MotOnTime = 8;        		  // 300
	sMotH2OCtrl.H2ODlyOn = 8; 			  //90
	sMotH2OCtrl.H2OOnTime = 8;            		  //30  
	gpio_MotH2OCtrl();
	sMotH2OCtrl.MotDlyOn = MOT_REV_DLY_ON_TIME_2;     //4
	sMotH2OCtrl.MotOnTime = 8;        		  //300
	sMotH2OCtrl.H2ODlyOn = MOT_REV_H2O_DLY_ON_TIME_2; //4
	sMotH2OCtrl.H2OOnTime = 8;            		  //30
	gpio_MotH2OCtrl();
	gpio_SysIdleInit();
	}

}



/**
  * @}
  */

/**
  * @}
  */
