/********************************************************************************
  * File Name          : timer.c
  * Description        : This source file contains the Timer and PWM code
  *******************************************************************************/

/* ============================== Includes ============================== */
#include "main.h"
#include "gpio.h"
#include "timer.h"

/* =========================== Private Defines =========================== */
#define SYS_TICK_VAL						(100 - 1)

#define TIM3_TICK								0.001
#define TIM3_RESOLUTION					0.00001
#define TIM3_PRESCALER					(uint16_t)(((double)SystemCoreClock * TIM3_RESOLUTION) - 1.0)
#define TIM3_PERIOD							(uint16_t)((TIM3_TICK / TIM3_RESOLUTION) - 1.0)

#define ONE_SEC_VAL							(uint16_t)(1 / TIM3_TICK)
#define ONE_MIN_VAL							60
#define ONE_HR_VAL							60
#define ONE_DAY_VAL							24


/* ========================== Private Variables ========================== */
uint32_t SysTickDlyCnt = 0;
uint16_t LidSwChkTimeCnt = 0;

TimeTypeStruct sTime;


/* ===================== Private Function Prototypes ====================== */


/* ============================= Functions ================================ */
void time_SysTickConfig(void)
{
	if (SysTick_Config(SystemCoreClock / SYS_TICK_VAL))
	{
		while(1);
	}
}

void time_SysTickDec(void)
{
  if (SysTickDlyCnt > 0)
  {
  	SysTickDlyCnt--;
  }
}

void time_SysTickDelay(uint32_t delay)
{
	if (sSysFlags.LidSwFlag == RESET)
	{
		SysTickDlyCnt = delay;
		while (SysTickDlyCnt != 0)
		{
			gpio_LidSwChk();
			if (sSysFlags.LidSwFlag == SET)
			{
				break;
			}
		}
	}
}

void tim3_TimerConfig(void)
{
  TIM_TimeBaseInitTypeDef	TIM3_InitStruct;
  NVIC_InitTypeDef NVIC_InitStruct;

	RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM3, ENABLE);

	TIM3_InitStruct.TIM_RepetitionCounter = ENABLE;
	TIM3_InitStruct.TIM_ClockDivision = TIM_CKD_DIV1;
	TIM3_InitStruct.TIM_CounterMode = TIM_CounterMode_Up;
	TIM3_InitStruct.TIM_Period = TIM3_PERIOD;
	TIM_TimeBaseInit(TIM3, &TIM3_InitStruct);

	TIM_PrescalerConfig(TIM3, TIM3_PRESCALER, TIM_PSCReloadMode_Immediate);

	NVIC_InitStruct.NVIC_IRQChannel = TIM3_IRQn;
	NVIC_InitStruct.NVIC_IRQChannelPriority = 0;
	NVIC_InitStruct.NVIC_IRQChannelCmd = ENABLE;
	NVIC_Init(&NVIC_InitStruct);

  TIM_ITConfig(TIM3, TIM_IT_Update, ENABLE);
  TIM_Cmd(TIM3, ENABLE);
}

void tim3_RtcUpdate(void)
{
	sTime.Time_mSec++;
	LidSwChkTimeCnt++;

	if (sTime.Time_mSec >= ONE_SEC_VAL)
	{
		sTime.Time_mSec = 0;
		sTime.Time_Sec++;
		PwrOnDlyCnt++;
		MotFwdOnCnt++;
		MotFwdDlyOnCnt++;
		MotRevOnCnt++;
		MotRevDlyOnCnt++;
		H2OOnCnt++;
		H2ODlyOnCnt++;
		H2OOnTimeoutCnt++;

		if (sTime.Time_Sec >= ONE_MIN_VAL)
		{
			sTime.Time_Sec = 0;
			sTime.Time_Min++;
			if (sTime.Time_Min >= ONE_HR_VAL)
			{
				sTime.Time_Min = 0;
				sTime.Time_Hr++;

				if (sTime.Time_Hr >= ONE_DAY_VAL)
				{
					sTime.Time_Hr = 0;
				}
			}
		}
	}
}


/**
  * @}
  */

/**
  * @}
  */
